<xsl:stylesheet
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:fn="fn"
    xmlns:pl="http://product-live.com"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    version="3.0"
    exclude-result-prefixes="xs fn pl">
    
    <xsl:output method="xml" indent="yes" encoding="UTF-8" /> 

    <!-- 
        This xslt do a differential between a new items file and an old items file
        It only export item when
        - The item exist in the new file but not in the old file
        - The item exist in the new file and in the old file, but at least on of the values of Identifiers, Classifications, or Fields is different
     -->

    <xsl:param name="oldItemsUrl" select="'./old-items.xml'" />
    <xsl:param name="groupingIdentifierKey" select="'ean'" />

    <!-- 
        Function used to merge Identifiers or Classifications or Fields
        @newAttribute an Identifier or Classification or Field²
        @oldAttribute an Identifier or Classification or Field
        return the Identifier or Classification or Field only if it's different
     -->
    <xsl:function name="pl:merge-attributes" as="item()*">
        <xsl:param name="newAttributes" as="item()*"/>
        <xsl:param name="oldAttributes" as="item()*"/>
        <xsl:merge>
            <xsl:merge-source name="new-attribute" select="$newAttributes" sort-before-merge="yes">
                <xsl:merge-key select="@key"/>
            </xsl:merge-source>
            <xsl:merge-source name="old-attribute" select="$oldAttributes" sort-before-merge="yes">
                <xsl:merge-key select="@key"/>
            </xsl:merge-source>
            <xsl:merge-action>
                <xsl:variable name="newAttribute" select="current-merge-group('new-attribute')" />
                <xsl:variable name="oldAttribute" select="current-merge-group('old-attribute')" />
                <xsl:choose>
                    <xsl:when test="$newAttribute and $oldAttribute">
                        <xsl:choose>
                            <!-- Test if it's a multiple select (fields with multiple select have can have multiple entries) -->
                            <xsl:when test="count($newAttribute) &gt; 1 or count($oldAttribute) &gt; 1">
                                <xsl:variable name="newAttributeString">
                                    <xsl:for-each select="$newAttribute">
                                        <xsl:sort select="." order="ascending" data-type="text" />
                                        <xsl:value-of select="concat(., @quantity, @comments)"/>
                                    </xsl:for-each>
                                </xsl:variable>
                                <xsl:variable name="oldAttributeString">
                                    <xsl:for-each select="$oldAttribute">
                                        <xsl:sort select="." order="ascending" data-type="text" />
                                        <xsl:value-of select="concat(., @quantity, @comments)"/>
                                    </xsl:for-each>
                                </xsl:variable>
                                <xsl:if test="$newAttributeString != $oldAttributeString">
                                    <xsl:copy-of select="$newAttribute"/>
                                </xsl:if>
                            </xsl:when>
                            <!-- Then if it is not a multiple select, compare values -->
                            <xsl:when test="$newAttribute!=$oldAttribute">
                                <xsl:copy-of select="$newAttribute"/>
                            </xsl:when>
                        </xsl:choose>                        
                    </xsl:when>
                    <xsl:when test="$newAttribute and not($oldAttribute)">
                        <xsl:copy-of select="$newAttribute"/>
                    </xsl:when>
                </xsl:choose>
            </xsl:merge-action>
        </xsl:merge>
    </xsl:function>

    <!-- 
        Return an item only if there is a difference on identifiers or classifications or fields or partition
        @newItem a Product-Live item
        @oldItem a Product-Live item
        return an array of Product-Live item or null
     -->
    <xsl:function name="pl:merge-two-items" as="item()?">
        <xsl:param name="newItem" as="item()"/>
        <xsl:param name="oldItem" as="item()"/>
        <xsl:variable name="identifiers" select="pl:merge-attributes($newItem/Identifier[@key!=$groupingIdentifierKey], $oldItem/Identifier[@key!=$groupingIdentifierKey])" />
        <xsl:variable name="classifications" select="pl:merge-attributes($newItem/Classification, $oldItem/Classification)" />
        <xsl:variable name="fields" select="pl:merge-attributes($newItem/Field, $oldItem/Field)" />
        <xsl:if test="$identifiers or $classifications or $fields or $newItem/@partition!=$oldItem/@partition">
            <Item partition="{$newItem/@partition}">
                <Identifier key="{$groupingIdentifierKey}"><xsl:value-of select="$newItem/Identifier[@key=$groupingIdentifierKey]"/></Identifier>
                <xsl:copy-of select="$identifiers"/>
                <xsl:copy-of select="$classifications"/>
                <xsl:copy-of select="$fields"/>
            </Item>
        </xsl:if>
    </xsl:function>
    
    <xsl:template match="/">
        <xsl:variable name="newItems" select="/" />
        <xsl:variable name="oldItems" select="doc($oldItemsUrl)" />
        <xsl:variable name="newTableKey" select="$newItems/Table/@key" />
        <xsl:variable name="oldTableKey" select="$oldItems/Table/@key" />
        <xsl:if test="$newTableKey != $oldTableKey">
            <xsl:message terminate="yes">Error: tables does not have the same table key. New Table has '<xsl:value-of select="$newTableKey"/>' but old Table has '<xsl:value-of select="$oldTableKey"/>'</xsl:message>
        </xsl:if>
        <Table key="{$newTableKey}">
            <Items>
                <xsl:merge>
                    <xsl:merge-source name="new-item" select="/Table/Items/Item" sort-before-merge="yes">
                        <xsl:merge-key select="Identifier[@key=$groupingIdentifierKey]/string()"/>
                    </xsl:merge-source>
                    <xsl:merge-source name="old-item" select="$oldItems/Table/Items/Item" sort-before-merge="yes">
                        <xsl:merge-key select="Identifier[@key=$groupingIdentifierKey]/string()"/>
                    </xsl:merge-source>
                    <xsl:merge-action>
                        <xsl:variable name="newItem" select="current-merge-group('new-item')" />
                        <xsl:variable name="oldItem" select="current-merge-group('old-item')" />
                        <xsl:choose>
                            <xsl:when test="$newItem and $oldItem">
                                <xsl:copy-of select="pl:merge-two-items($newItem, $oldItem)"/>
                            </xsl:when>
                            <xsl:when test="$newItem and not($oldItem)">
                                <xsl:copy-of select="$newItem"/>
                            </xsl:when>
                        </xsl:choose>
                    </xsl:merge-action>
                </xsl:merge>
            </Items>
        </Table>
    </xsl:template>
</xsl:stylesheet>